"use strict";

String.prototype.replaceAll = function (search, replacement) {
    var target = this;
    return target.replace(new RegExp(search, 'g'), replacement);
};

function CalculateVolume(isMusic) {
    let rootScope = GetRootScope();
    if (rootScope == null || rootScope.options == null) return;

    let volume = isMusic
        ? rootScope.options.musicVolume
        : rootScope.options.sfxVolume;

    let newVolume = volume * rootScope.options.masterVolume / 100;
    return newVolume;
}

function HideMouseoverEvents() {
    $('.tooltip, .popover').remove();
}

function SetFullScreen(fullscreen) {
    if (Remote != null) {
        Remote.getCurrentWindow().setFullScreen(fullscreen);
        SetZoom();
    }
}

function PushAsStack(array, obj, stackSize) {
    while (array.length >= stackSize) {
        array.shift();
    }
    array.push(obj);
}

function CalculateEmploymentDuration(employee, currentDate) {
    if (employee == null || employee.hired == null) return 0;
    return CalculateDaysSince(currentDate, employee.hired);
}

function CalculateDaysSince(baseDate, compareDate) {
    return Math.round(moment.duration(new Date(baseDate) - new Date(compareDate)).asDays());
}

function CalculateDaysSinceLastRaise(employee, currentDate) {
    if (!employee) return false;

    let startDate = employee.lastSalaryRaise || employee.hired;
    let durationInDays = moment.duration(new Date(currentDate) - new Date(startDate)).asDays();

    return durationInDays;
}

function GetRevenueAsMonthly(revenuePerUserPerHour) {
    return revenuePerUserPerHour * 24 * 30;
}

function ApplyBonuses(workstation, value, officeBonus) {
    // Add office bonus
    value = value * (100 + officeBonus) / 100;

    // Add manager bonus
    value += GetManagerBonus(workstation);

    // Add mood penality
    value += GetMoodPenalty(workstation);

    return value;
}

function ClearHoveredCells() {
    $('.grid .cell.hover').removeClass('hover');
}

function GetZone(gridSize, width, height, tileId, orientation) {
    let cells = [];

    let currentId = tileId;
    let lastLine = GetLineNumber(tileId, gridSize);

    let originalWidth = width;
    let originalHeight = height;

    if (orientation % 2 == 0) {
        width = originalHeight;
        height = originalWidth;
    }

    for (let x = 1; x <= height; x++) {
        for (let y = 0; y < width; y++) {
            if (GetLineNumber(currentId, gridSize) == lastLine && currentId >= 0) {
                cells.push(currentId);
            } else {
                return null;
            }

            lastLine = GetLineNumber(currentId, gridSize);
            currentId -= 1;
        }

        currentId = tileId - (gridSize * x);
        lastLine = GetLineNumber(currentId, gridSize);
    }

    return cells;
}

function GetLineNumber(id, size) {
    return Math.ceil((id + 1) / size);
}

function GetCellCoordinates(cellId, placementOffset) {
    let position = $('#' + cellId).position();
    if (position == null) return {top: 0, left: 0};
    return {top: position.top + placementOffset.top, left: position.left + placementOffset.left};
}

var _RootScope = null;
function GetRootScope() {
    if (_RootScope == null) {
        _RootScope = angular.element(document.getElementsByTagName('body')).scope();
        if (_RootScope.$parent != null)
            _RootScope = _RootScope.$parent;
    }

    return _RootScope;
}

function IsDialogOpen() {
    let visibleDialogs = $('.dialog:visible');
    return visibleDialogs.length > 0;
}

function SetZoom() {
    let zoomLevel = GetRootScope().options.zoomLevel;
    let office = $('office');
    if (zoomLevel == null || !office.length) return;

    let minimumZoom = Helpers.CalculateMinimumZoom();
    let zoom = 0;
    let multiplier = (100 - Helpers.CalculateMinimumZoom()) / 5;
    switch (zoomLevel.value) {
        case 0:
            zoom = minimumZoom;
            break;
        case 1:
            zoom = minimumZoom + (multiplier * 1);
            break;
        case 2:
            zoom = minimumZoom + (multiplier * 2);
    }

    $('.background').css('zoom', zoom + "%");

    // Center the view
    let left = (office[0].scrollWidth - office[0].clientWidth) / 2;
    let top = (office[0].scrollHeight - office[0].clientHeight) / 2;
    office.scrollLeft(left);
    office.scrollTop(top);
}

function CalculateWorkstationProductivity(workstation) {
    if (workstation.employee == null) return null;
    let result = ApplyBonuses(workstation, workstation.employee.speed, GetRootScope().officeBonus);
    return result;
}


function GetDateDiffInDays(date) {
    let start = moment(GetRootScope().settings.started).hour(0).minute(0);
    let end = moment(date).hour(0).minute(0);
    return Math.round(moment.duration(end - start).asDays() + 1);
}

function GetDateDiffInHours(date) {
    return Math.round(moment.duration(new Date(date) - GetRootScope().settings.date).asHours());
}

function RemoveEmployee(employee, resigned) {
    let $rootScope = GetRootScope();
    let workstation = $rootScope.settings.office.workstations.find(x => x.employee != null && x.employee.id == employee.id);

    let allWorkstations = $rootScope.settings.office.workstations;
    let managerWorkstation = allWorkstations.find(x =>
    x.employee != null &&
    x.employee.employees != null &&
    x.employee.employees.some(employeeId => employeeId == employee.id));

    if (managerWorkstation != null) {
        _.remove(managerWorkstation.employee.employees, x => x == employee.id);
    }

    if(resigned) {
        $rootScope.settings.resignedEmployees.push(employee);
    } else {
        $rootScope.settings.firedEmployees.push(employee);
    }

    if (workstation != null) {
        workstation.employee = null;
    } else {
        _.remove($rootScope.settings.unassignedEmployees, x => x.id == employee.id);
    }
}

function GetManagerWorkstationByEmployeeId(employeeId) {
    return GetRootScope().settings.office.workstations.find(ws => ws.employee != null && ws.employee.employees.some(x => x == employeeId));
}

function GetManagerBonus(workstation) {
    if (workstation.employee == null) return 0;

    let managerWorkstation = GetManagerWorkstationByEmployeeId(workstation.employee.id);
    if (managerWorkstation == null) return 0;

    if (managerWorkstation.employee.vacationHoursLeft > 0) return 0; // Is on vacation

    let managerProductivity = CalculateWorkstationProductivity(managerWorkstation);
    return managerProductivity / managerWorkstation.employee.employees.length;
}

function GetMoodPenalty(workstation) {
    if (workstation.employee == null) return 0;
    return -(100 - workstation.employee.mood);
}

function GetJobCompletionPercent(job) {
    let percentages = [];
    job.requirements.forEach(requirement => {
        percentages.push(Math.min(requirement.amountCompleted * 100 / requirement.amount, 100));
    });

    let result = _.sum(percentages) / percentages.length;
    return result;
}


function HasEmployeeHrManager(employee) {
    if (employee.employeeTypeName == EmployeeTypeNames.HrManager) return true;
    let managerWs = GetManagerWorkstationByEmployeeId(employee.id);
    if (managerWs != null && managerWs.employee != null) {
        if (managerWs.employee.employeeTypeName == EmployeeTypeNames.HrManager) return true;

        let levelTwoManagerWs = GetManagerWorkstationByEmployeeId(managerWs.employee.id);
        if (levelTwoManagerWs != null && levelTwoManagerWs.employee.employeeTypeName == EmployeeTypeNames.HrManager) return true;
    }

    return false;
}

function ToggleLoader(show) {
    let elm = $('#loading-screen');
    if (!show) {
        elm.addClass('animated fadeOut').on('webkitAnimationEnd', function () {
            $(this).removeClass('animated fadeOut').hide();
        });
    } else {
        elm.show();
    }
}

function IsJobCompleted(job) {
    return job.requirements.every(x => parseFloat(x.amountCompleted.toFixed(1)) >= x.amount);
}

let DeliverContract = (contract) => {
    let $rootScope = GetRootScope();
    contract.completed = true;
    contract.status = ContractStatuses.Won;
    contract.completionDay = GetDateDiffInDays($rootScope.settings.date);

    let product = CompetitorProducts.find(x => x.id == contract.productId);
    contract.pastDueFee = Helpers.CalculatePastDueFee(contract);
    contract.totalPayout = (contract.price - contract.pastDueFee);
    $rootScope.settings.balance += contract.totalPayout;
    $rootScope.addTransaction(Helpers.GetLocalized('transaction_contract_completed', {
        name: product.name,
        number: contract.number
    }), contract.totalPayout);

    let productionHours = _.sum(Object.keys(contract.requirements).map(componentName => Helpers.CalculateComponentProductionHours(Components.find(x => x.name == componentName)) * contract.requirements[componentName]));

    // Removed XP for calculation below. Sorry for the mess.
    contract.xpEarned = Math.round(Math.round((productionHours * contract.pricePerHour) / 50) * 3);

    $rootScope.settings.xp += contract.xpEarned;

    // Grab components
    Object.keys(contract.requirements).forEach(componentName => {
        $rootScope.settings.inventory[componentName] -= contract.requirements[componentName];
    });

    $rootScope.settings.contractReportId = contract.id;
    Helpers.DetachContractFromEmployee(contract);
    $rootScope.settings.maxContractHours += 2;
    $rootScope.$broadcast(GameEvents.ContractChange);
    PlaySound(Sounds.money);
};

let GetSkillByName = (name) => {
    return Skills.find(x => x.name == name);
};

let Helpers = {
    GetAvailableJobs: () => {
        return GetRootScope().settings.contracts.filter(x => x.status == ContractStatuses.NotSet);
    },
    GetDateDiffInHours: (date) => {
        return GetDateDiffInHours(date);
    },
    GetSkillByName: GetSkillByName,
    CalculateProductivityBySkill: (workstation, skillName) => {
        let employee = workstation.employee;
        let totalProductivity = CalculateWorkstationProductivity(workstation);
        return totalProductivity;
    },
    GetXpByTier: tier => {
        let tiers = {
            1: 0,
            2: 200,
            3: 800,
            4: 2500,
            5: 5000,
            6: 8000,
            7: 15000,
            8: 28000,
            9: 50000,
            10: 90000,
            11: 200000,
            12: 300000,
            13: 340000,
            14: 380000,
            15: 360000,
            16: 370000,
            17: 380000,
            18: 390000,
            19: 400000,
            20: 500000,
        };
        return tiers[tier];
    },
    CalculateCompanyTier: (repPoints) => {
        if (repPoints == null) {
            repPoints = GetRootScope().settings.xp;
        }
        let tier = null;
        let i = 1;
        do {
            let current = Helpers.GetXpByTier(i);
            let next = Helpers.GetXpByTier(i + 1);
            if (current == null || next == null) tier = i;
            if (_.inRange(repPoints, current, next)) {
                tier = i;
            } else {
                i++;
            }
        } while (tier == null);

        return tier;
    },
    CalculateXpForTier: (level) => {
        return (level / 0.05) ^ 2;
    },
    GetLast30DaysContracts: () => {
        let today = GetDateDiffInDays(GetRootScope().settings.date);
        return GetRootScope().settings.contracts.filter(x => x.completed && x.winnerOffer != null && x.completionDay >= (today - 30))
    },
    CalculateBaseSalary: (employeeTypeName, level, speed) => {
        let salary = 0;

        switch (level) {
            case EmployeeLevels.Beginner:
                salary = 2700;
                break;
            case EmployeeLevels.Intermediate:
                salary = 4500;
                break;
            case EmployeeLevels.Expert:
                salary = 6000;
                break;
        }

        salary += (speed - 50) * 30;

        return salary;
    },
    GetPastDueFeeHourly: (urgency) => {
        let perHour = 100;
        let result = 0;

        switch (urgency) {
            case Priorities.Low:
                result = perHour;
                break;
            case Priorities.Medium:
                result = perHour * 1.5;
                break;
            case Priorities.High:
                result = perHour * 3;
                break;
        }

        return result;
    },
    CalculatePastDueFee: contract => {
        let hours = contract.hoursLeft;
        if (hours < 0) {
            let perHour = Helpers.GetPastDueFeeHourly(contract.urgency);
            return Math.abs(hours) * perHour;
        }

        return 0;
    },
    GetDeadlineText: hoursLeft => {
        if (hoursLeft > 0) {
            return Helpers.GetLocalized('hours_left', {hours: Math.abs(hoursLeft)});
        } else {
            return Helpers.GetLocalized('hours_late', {hours: Math.abs(hoursLeft)});
        }
    },
    Kformatter: num => {
        // http://stackoverflow.com/questions/9461621/how-to-format-a-number-as-2-5k-if-a-thousand-or-more-otherwise-900-in-javascrip
        return num > 999 ? (num / 1000).toFixed(0) + 'K' : num
    },
    CalculateComponentProductionHours: component => {
        if (component.type == ComponentTypes.Component) {
            return component.produceHours;
        }

        let hours = 0;
        Object.keys(component.requirements).forEach(requirementName => {
            let requirementComponent = Components.find(x => x.name == requirementName);
            hours += Helpers.CalculateComponentProductionHours(requirementComponent);
        });

        return hours;
    },
    GetBaseComponentPopoverText: component => {
        let lines = [];
        if (component.requirements != null) {
            Object.keys(component.requirements).forEach(componentName => {
                let requirementComponent = Components.find(x => x.name == componentName);
                let amount = GetRootScope().settings.inventory[componentName] || 0;
                let amountAvailable = amount >= component.requirements[componentName];
                let amountDisplay = amountAvailable ? component.requirements[componentName] : `${amount}/${component.requirements[componentName]}`;

                lines.push(`<div class="module-popover-requirement ${ amountAvailable ? null : "color-red"}"><img src="${requirementComponent.icon}"><span class="amount">${ amountDisplay }</span><span class="name">${ Helpers.GetLocalized(requirementComponent.name) }</span></div>`)
            });
        } else {
            lines.push(`<icon class="${component.employeeType}">${ Helpers.GetLocalized(component.employeeType) }</icon>`);
            lines.push(`Workload: <strong>${ component.produceHours } hours</strong>`);
        }

        return `<div class="text-center">${_.join(lines, ' ')}</div><hr>`;
    },
    Clone: (obj, keepId) => {
        let newObj = JSON.parse(JSON.stringify(obj));
        if (newObj.id != null && (keepId == null || keepId == false)) {
            newObj.id = chance.guid();
        }
        return newObj;
    },
    GetLast30DaysIncomeTransactions: () => {
        let settings = GetRootScope().settings;
        let currentDay = GetDateDiffInDays(settings.date);
        let orderedTransactions = _.orderBy(settings.transactions, 'order', 'desc');
        return orderedTransactions.filter(x => x.amount > 0 && x.day >= currentDay - 30);
    },
    GetDevelopmentTask: component => {
        let produceHours = Helpers.CalculateComponentProductionHours(component);

        return {
            component: component,
            totalMinutes: produceHours * 60,
            completedMinutes: 0,
            state: Enums.TaskStates.Running
        };
    },
    CalculateProducedMinutes: employeeStat => {
        return employeeStat.productivity * 1 / 100;
    },
    CalculateMaxInCharge: employee => {
        switch (employee.level) {
            case EmployeeLevels.Beginner:
                return 3;
            case EmployeeLevels.Intermediate:
                return 5;
            case EmployeeLevels.Expert:
                return 8;
        }
    },
    GetAllEmployees: (includeUnassignedEmployees) => {
        let result = GetRootScope().settings.office.workstations.filter(x => x.employee != null).map(x => x.employee);
        if (includeUnassignedEmployees) {
            result = _.union(result, GetRootScope().settings.unassignedEmployees);
        }

        return result;
    },
    CalculateWer: () => {
        let $rootScope = GetRootScope();
        let officeBonus = $rootScope.officeBonus;
        let benefitBonuses = _.sumBy($rootScope.settings.activatedBenefits.map(benefitId => Database.benefits.find(benefit => benefit.id == benefitId)), x => x.bonus);

        return {officeBonus: officeBonus, benefitBonuses: benefitBonuses, total: officeBonus + benefitBonuses};
    },
    CalculateBenefitCost: benefit => {
        let employees = Helpers.GetAllEmployees();
        let perEmployee = employees.length * benefit.pricePerEmployee;

        let daily = perEmployee + Math.round(benefit.fixedPrice / 30);

        return {
            daily: daily,
            monthly: daily * 30
        }
    },
    LoadSettingsFromDisk: (filename, cb) => {
        Remote.app.loadFile(filename, data => {
            let settings = JSON.parse(data);
            if (settings == null) {
                cb(null)
            } else {
                settings.date = new Date(settings.date); // JSON does not support dates
                settings.started = settings.started == false ? false : new Date(settings.started);
                settings.paused = true;
                cb(settings);
            }
        });
    },
    LoadGame: ($rootScope, filename, cb) => {
        Helpers.LoadSettingsFromDisk(filename, settings => {
            if (settings == null) {
                cb(null);
                return;
            }

            Helpers.PrepareSavegameCompatibility(settings);

            $rootScope.settings = settings;
            Helpers.ResetEngine();
            SetZoom();

            wallhack.sendEvent("load_game", `Company Name: ${ settings.companyName }`);

            cb(settings);
        });
    },
    ResetEngine: () => {
        let $rootScope = GetRootScope();
        ProfitPerHourHistory = [];

        IndicatorHistory = {};

        Game.Lifecycle = new Lifecycle($rootScope);
        Game.Lifecycle._update(0);
        Game.Lifecycle.PauseTime(true, true);
        $rootScope.financeData = null;
        $rootScope.updateFinanceData();

        _.toArray(GameEvents).forEach(gameEvent => {
            $rootScope.$broadcast(gameEvent);
        });
        if (!$rootScope.$$phase) $rootScope.$digest();
    },
    GetSaveGameIdentifer: saveGameId => {
        return `sg_${saveGameId}.dton`;
    },
    GetOptionsIdentifer: () => {
        return `options.dton`;
    },
    ConvertRequirementsIntoStacks: (requirements) => {
        let $rootScope = GetRootScope();
        return Object.keys(requirements).map(componentName => {
            return {
                component: Components.find(y => y.name == componentName),
                amount: requirements[componentName],
                isAvailableInInventory: requirements[componentName] <= $rootScope.settings.inventory[componentName]
            }
        });
    },
    ConvertComponentsIntoStacks: (components, selectedComponentIds) => {
        if (selectedComponentIds == null) selectedComponentIds = [];

        let remainingComponents = components.filter(x => !selectedComponentIds.includes(x.id));

        let stacks = _.orderBy(_.toArray(_.groupBy(remainingComponents.filter(x => x != null), x => x.name))
            .map(x => {
                return {component: x[0], amount: x.length}
            }).filter(x => x.amount != 0), x => x.component.name);

        stacks.forEach(stack => {
            let componentsInInventory = GetRootScope().settings.inventory.filter(x => x.name == stack.component.name);
            stack.isAvailableInInventory = componentsInInventory.length >= stack.amount;
        });

        return stacks;
    },
    GetRequiredComponentsForProduction: component => {
        let components = [];
        if (component.type == ComponentTypes.Component) {
            return component;
        }

        // module
        Object.keys(component.requirements).forEach(requirementComponentName => {
            let requirementComponent = Components.find(x => x.name == requirementComponentName);

            if (requirementComponent.type == ComponentTypes.Component) {
                components.push(requirementComponent);
            } else {
                let generatedComponents = Helpers.GetRequiredComponentsForProduction(requirementComponent);
                components = _.union(components, generatedComponents, [requirementComponent]);
            }
        });

        return components;
    },
    GetTierForEmployeeLevels: () => {
        let result = {};
        result[EmployeeLevels.Beginner] = 1;
        result[EmployeeLevels.Intermediate] = 5;
        result[EmployeeLevels.Expert] = 10;
        return result;
    },
    ConsoleInfo(message, type) {
        message = `[${moment().format('HH:mm:ss')}] ${message}`;
        switch (type) {
            case "warning":
                console.warn(message);
                break;
            default:
                console.info(message);
                break;
        }
    },
    SubtractUppercaseCharacters: value => {
        let result = "";
        _.toArray(value).forEach(char => {
            if (char == char.toUpperCase()) {
                result += char;
            }
        });

        if (result.length < 2) {
            result = value.substr(0, 2).toUpperCase();
        }

        return result;
    },
    GetItemPopover: (item, showRelativeBonus) => {
        let result = [];

        let bonus = item.bonus;
        if (showRelativeBonus) {
            bonus = Helpers.CalculateBonusByItem(item.name).bonusPerItem;
        }

        if (item.tier > GetRootScope().companyTier) {
            result.push(`<strong>${ Helpers.GetLocalized('unlocked_at_tier', {tier: item.tier}) }</strong>`);
        }
        result.push(Helpers.GetLocalized('office_bonus_amount', {bonus: `${bonus}%${ showRelativeBonus && item.bonus != bonus ? `<span class="small color-secondary-grey">(${ item.bonus }%)</span>` : '' }`}));
        result.push(Helpers.GetLocalized('price_amount', {amount: numeral(item.price).format(Configuration.CURRENCY_FORMAT)}));

        return `<div class="text-center">${ _.join(result, '<br />')}</div>`;
    },
    CalculateMinimumZoom: () => {
        let ground = $('.background .office');
        let minimumZoom = Math.ceil(Math.max(
            (window.innerHeight * 100 / ground.height()),
            (window.innerWidth * 100 / ground.width())
        ));

        return isFinite(minimumZoom) ? minimumZoom : 20;
    },
    DetachContractFromEmployee: contract => {
        let employees = Helpers.GetAllEmployees(true).filter(x => x.employeeTypeName == EmployeeTypeNames.SalesExecutive);

        let employee = employees.find(employee => employee.task != null
        && (employee.task.contractId == contract.id || (employee.task.contract != null && employee.task.contract.id == contract.id)));

        if (employee != null && employee.task != null) {
            employee.task.contractId = null;
            employee.task.completedMinutes = 0;

            if (employee.task.autoRepeat) {
                employee.task.contract = Generators.GenerateContract(employee.task.contractType, employee.task.size);
                employee.task.state = Enums.TaskStates.Running;
            } else {
                employee.task.state = Enums.TaskStates.Stopped;
            }
        }

        GetRootScope().$broadcast(GameEvents.WorkstationChange);
    },
    CancelContract: contract => {
        Helpers.DetachContractFromEmployee(contract);
        contract.completed = true;
        contract.status = ContractStatuses.Cancelled;
        let $rootScope = GetRootScope();
        let cancellationFee = Helpers.CalculateCancellationFee(contract);
        $rootScope.settings.balance += -cancellationFee;
        $rootScope.addTransaction(`Contract ${ contract.number }: Cancellation Fee`, -cancellationFee);
        $rootScope.settings.maxContractHours--;
        $rootScope.$broadcast(GameEvents.ContractChange);
    },
    CalculateCancellationFee: contract => {
        return contract.price / 3;
    },
    GetResearchPrice: component => {
        if (component.researchPrice) {
            return component.researchPrice
        } else {
            let basePrice = 0;

            switch (component.employeeLevel) {
                case EmployeeLevels.Beginner:
                    basePrice += 1000;
                    break;
                case EmployeeLevels.Intermediate:
                    basePrice += 3000;
                    break;
                case EmployeeLevels.Expert:
                    basePrice += 5000;
                    break;
            }

            if (component.type == ComponentTypes.Module) {
                basePrice = basePrice * 1.2;
            }

            return basePrice;
        }
    },

    GetIndexByObject: (level) => {
        return _.toArray(EmployeeLevels).indexOf(level);
    },
    GetRequiredCusByComponents: components => {
        let produceHours = _.sum(components.map(x => Helpers.CalculateComponentProductionHours(x)));
        return Math.round(produceHours / 2);
    },
    GetExponentialGrowth: (level, size, increaseRate) => {
        if (increaseRate <= 0) return size;
        if (level == 0) return 0;

        let result = ((level * increaseRate) * (level * increaseRate)) * size;
        return Math.max(result, size);
    },
    PrepareSavegameCompatibility: settings => {
        let currentVersion = `${Configuration.BETA_VERSION}.${Configuration.BETA_SUBVERSION}`;
        settings.lastVersion = settings.lastVersion || '8.0';
        settings.resignedEmployees = settings.resignedEmployees || [];

        if (Helpers.VersionCompare('9.0', settings.lastVersion) > 0) {
            settings.maxContractHours = settings.maxContractHours || 5;
            settings.notifications = settings.notifications || [];
            settings.activatedBenefits = settings.activatedBenefits || [];
            settings.contracts = [];

            for (let i = 0; i < settings.inventory.length; i++) {
                let newOne = Components.find(x => x.name == settings.inventory[i].name);
                if (newOne != null) {
                    settings.inventory[i] = Helpers.Clone(newOne);
                }
            }

            _.remove(settings.inventory, x => x.name == 'ResponsiveComponent');
            _.remove(settings.inventory, x => x.name == 'WebDesignComponent');
            _.remove(settings.inventory, x => x.name == 'WebFrontendModule');

            settings.candidates.forEach(candidate => {
                if (candidate.requiredWer == null) {
                    candidate.requiredWer = candidate.requiredDesirability;
                }
            });

            settings.products = [];
        }

        if (Helpers.VersionCompare('10.0', settings.lastVersion) > 0) {
        }

        settings.researchInventory = settings.researchInventory || Helpers.Clone(Database.newGameSettings.researchInventory);

        if (settings.floors != null) {
            let itemPrice = _.sum(settings.floors[0].items.map(x => x.item.price));
            let workstationPrice = settings.floors[0].workstations.length * 5500;
            settings.balance += (itemPrice + workstationPrice);

            settings.office = {
                buildingName: 'MediumBuilding',
                items: [],
                workstations: []
            };

            settings.unassignedEmployees = [
                ...settings.unassignedEmployees,
                ...settings.floors[0].workstations.map(x => x.employee).filter(x => x != null)];

            delete settings.floors;
        }

        if (_.isArray(settings.inventory)) {
            settings.inventory = {};
        }

        settings.office.workstations.forEach(workstation => {
            if (workstation.bonus == null) {
                workstation.bonus = 0;
            }
            if (workstation.employee != null) {
                if (workstation.employee.task != null && workstation.employee.task.state == null) {
                    workstation.employee.task = null;
                }
            }
        });

        // Remove in Beta 11
        if (settings.candidates.length > 600) {
            settings.candidates = [];
        }

        settings.mails.filter(x => x.id == null).forEach(mail => {
           mail.id = chance.guid();
        });

        settings.products.forEach(product => {
            product.activeMarketingPackages = product.activeMarketingPackages || [];
        });

        // Remove duplicated contracts (caused by Beta 9/10 bugs)
        _.flatten(_.toArray(_.groupBy(settings.contracts, x => x.id)).filter(x => x.length > 1)).forEach(contract => {
           _.remove(settings.contracts, x => x.id == contract.id);
        });

        settings.lastVersion = currentVersion;
    },
    PrepareOptionsCompatibility: options => {
        options.language = options.language || 'en';
        options.completedAchievements = options.completedAchievements || [];
    },
    VersionCompare: (v1, v2, options) => {
        let lexicographical = options && options.lexicographical,
            zeroExtend = options && options.zeroExtend,
            v1parts = v1.split('.'),
            v2parts = v2.split('.');

        function isValidPart(x) {
            return (lexicographical ? /^\d+[A-Za-z]*$/ : /^\d+$/).test(x);
        }

        if (!v1parts.every(isValidPart) || !v2parts.every(isValidPart)) {
            return NaN;
        }

        if (zeroExtend) {
            while (v1parts.length < v2parts.length) v1parts.push("0");
            while (v2parts.length < v1parts.length) v2parts.push("0");
        }

        if (!lexicographical) {
            v1parts = v1parts.map(Number);
            v2parts = v2parts.map(Number);
        }

        for (let i = 0; i < v1parts.length; ++i) {
            if (v2parts.length == i) {
                return 1;
            }

            if (v1parts[i] == v2parts[i]) {
                continue;
            }
            else if (v1parts[i] > v2parts[i]) {
                return 1;
            }
            else {
                return -1;
            }
        }

        if (v1parts.length != v2parts.length) {
            return -1;
        }

        return 0;
    },
    CalculateOfficeBonus: () => {
        let total = 0;
        let $rootScope = GetRootScope();

        _.uniq($rootScope.settings.office.items.map(x => x.item.name)).forEach(itemName => {
            let bonusInfo = Helpers.CalculateBonusByItem(itemName);
            total += bonusInfo.bonusPerItem * bonusInfo.count;
        });

        total += _.sum($rootScope.settings.office.workstations.map(x => x.bonus)); // Workstation bonus

        return parseFloat(total.toFixed(1));
    },
    CalculateBonusByItem: (itemName, previewPurchase) => {
        let $rootScope = GetRootScope();
        let items = $rootScope.settings.office.items.filter(x => x.item.name == itemName);
        let count = items.length;
        let exampleItem = Database.items.find(x => x.name == itemName);
        if (previewPurchase) count++;

        let bonusPerItem = exampleItem.bonus / (count > 1 ? Math.max((count * 0.45), 1) : 1);
        bonusPerItem = parseFloat(bonusPerItem.toFixed(2));

        return {bonusPerItem: bonusPerItem, count: count};
    },
    TakeCareOfResumingGame: () => {
        let $rootScope = GetRootScope();
        if ($rootScope.settings.resumeGameWhenClosed == true) {
            if ($rootScope.settings.contractReportId == null && $rootScope.settings.tierReport == null)
                $rootScope.settings.resumeGameWhenClosed = null;
            Game.Lifecycle.PauseTime(true, false);
        }
    },
    MovePropToIsolatedScope: ($rootScope, $scope, identifier) => {
        let watcher = $rootScope.$watch(identifier, value => {
            $scope[identifier] = value;
        });

        $scope.$on('$destroy', () => {
            watcher();
        });
    },
    RandomizeNumber: (number, differenceInPercentage, round) => {
        differenceInPercentage = differenceInPercentage / 100;
        let result = _.random(number * (1 - differenceInPercentage), number * (1 + differenceInPercentage));
        if (round == true) result = Math.round(result);
        return result;
    },
    RootScopeWatchAndDestroy: (localScope, identifier, action, deep) => {
        let watcher = GetRootScope().$watch(identifier, action, deep);
        localScope.$on('$destroy', () => watcher());
    },
    RootScopeWatchCollectionAndDestroy: (localScope, identifier, action) => {
        let watcher = GetRootScope().$watchCollection(identifier, action);
        localScope.$on('$destroy', () => watcher());
    },
    GetComponentsByEmployeeLevel: (employeeLevel) => {
        let employeeLevels = [EmployeeLevels.Beginner];
        switch (employeeLevel) {
            case EmployeeLevels.Intermediate:
                employeeLevels.push(EmployeeLevels.Intermediate);
                break;
            case EmployeeLevels.Expert:
                employeeLevels.push(EmployeeLevels.Intermediate);
                employeeLevels.push(EmployeeLevels.Expert);
                break;
        }

        return Components.filter(x => employeeLevels.includes(x.employeeLevel));
    },
    IsEmployeeLevelCorrect: (target, current) => {
        switch (target) {
            case EmployeeLevels.Beginner:
                return true;
                break;
            case EmployeeLevels.Intermediate:
                return [EmployeeLevels.Intermediate, EmployeeLevels.Expert].includes(current);
                break;
            case EmployeeLevels.Expert:
                return current == EmployeeLevels.Expert;
                break;
        }
    },
    GetCurrentEmployeeStat: () => {
        let $rootScope = GetRootScope();
        if ($rootScope.selectedWorkstation == null || $rootScope.selectedWorkstation.employee == null) return null;
        return Game.Lifecycle.EmployeeStats.find(x => x.employee.id == $rootScope.selectedWorkstation.employee.id);
    },
    GenerateUserSegment: (featureName, platformName, productTypeName) => {
        let productType = ProductTypes.find(x => x.name == productTypeName);
        let userbase = productType.users[platformName];
        let gainedUsers = userbase * 0.00001;

        return {
            featureName: featureName,
            platformName: platformName,
            usersLeft: gainedUsers,
            minutesLeft: 43200,
            gainPercentagePerMinute: 0.00013
        }
    },
    RemoveEmployeeFromWorkstation: workstation => {
        if (workstation.employee == null) return;

        let $rootScope = GetRootScope();

        // Detach from manager
        let managerWs = GetManagerWorkstationByEmployeeId(workstation.employee.id);
        if (managerWs != null) {
            _.remove(managerWs.employee.employees, employeeId => employeeId == workstation.employee.id);
            if(workstation.employee.task != null) {
                workstation.employee.task.autoRepeat = false;
            }
        }

        // If manager, remove all employees
        if (workstation.employee.employeeTypeName == EmployeeTypeNames.Manager) {
            workstation.employee.employees = [];
        }

        $rootScope.settings.unassignedEmployees.push(workstation.employee);
        workstation.employee = null;
        $rootScope.$broadcast(GameEvents.WorkstationChange);
        Game.Lifecycle._loadEmployeeStats();
    },
    GetLocalized: (key, data) => {
        let result = Language[key.toLowerCase()];
        if (result == null || result == "") result = ">:" + (EnglishLanguage[key.toLowerCase()] || `${key.toLowerCase()}`);

            if (data != null) {
        Object.keys(data).forEach(key => {
            result = result.replaceAll(`{${key}}`, data[key]);
        });
    }

    return result;
    },
    CalculateRequiredCuForProduct: product => {
        let cu = 0;
        product.features.forEach(featureInfo => {
            let feature = Features.find(x => x.name == featureInfo.featureName);
            cu += feature.requiredComputeUnit * featureInfo.level;
        });

        return cu;
    },
    CalculateTotalCuByProduct: product => {
        let cu = 1000;
        Servers.forEach(server => {
            if (product.servers[server.name] != null) {
                cu += product.servers[server.name] * server.computeUnit;
            }
        });

        return cu;
    },
    CalculateHostingExpenses: product => {
        let expenses = 375;
        Servers.forEach(server => {
            if (product.servers[server.name] != null) {
                expenses += (server.pricePerDay * product.servers[server.name]);
            }
        });

        return expenses;
    },
    FireEmployee: employee => {
        let $rootScope = GetRootScope();
        let message = `<p>${ Helpers.GetLocalized('confirm_dismiss') }</p>`;
        let contract = null;
        if (employee.contractId != null) {
            message += `<p class="color-red text-bold">${ Helpers.GetLocalized('confirm_dismiss_contract') }</p>`;
            contract = $rootScope.settings.contracts.find(x => x.id == employee.contractId);
        }

        $rootScope.confirm(Helpers.GetLocalized('are_you_sure'), message, () => {
            if (contract != null) {
                Helpers.CancelContract(contract);
            }

            RemoveEmployee(employee);

            // Decrease happiness of all other employees
            $rootScope.settings.office.workstations.forEach(ws => {
                if (ws.employee != null) {
                    ws.employee.mood -= 5;
                }
            });

            $rootScope.$broadcast(GameEvents.EmployeeChange);
        });
    },
    GetRequirementsForFeatureLevel: (feature, level) => {
        let requirements = Helpers.Clone(feature.requirements);
        Object.keys(requirements).forEach(componentName => {
            requirements[componentName] = requirements[componentName] * level;
        });

        return requirements;
    },
    LoadLanguage: (language, cb) => {
        let englishLanguage = Database.languages.find(x => x.key == 'en');
        $.getJSON(englishLanguage.path, function (response) {
            EnglishLanguage = response;
        }).fail((jqXHR, textStatus, errorThrown) => {
            console.error(`${language.path}: ${errorThrown}`);
        });

        $.getJSON(language.path, function (response) {
            Language = response;

            // Manipulate CSS localizations
            let sheet = _.last(window.document.styleSheets);
            let translation = Helpers.GetLocalized('locked');
            sheet.insertRule(`.tier-locked:after { content: '${ translation }' }`, sheet.cssRules.length);

            cb();
        }).fail((jqXHR, textStatus, errorThrown) => {
            console.error(`${language.path}: ${errorThrown}`);
        });
    },
    CalculateRequiredResearchPoints: (featureName, frameworkName) => {
        if (featureName != null) {
            let feature = Features.find(x => x.name == featureName);
            return Math.round(_.sum(Object.keys(feature.requirements).map(componentName => Helpers.CalculateComponentProductionHours(Components.find(x => x.name == componentName)) * feature.requirements[componentName])));
        }

        if (frameworkName != null) {
            let framework = Frameworks.find(x => x.name == frameworkName);
            return (framework.maxFeatures + framework.maxFeatureLevel) * 5;
        }

        return 0;
    },
    PrepareItemPath: (path, orientation) => {
        if (orientation == null) orientation = 3;
        return path.replaceAll('{orientation}', orientation);
    },
    GetEmployeeTypeCssClass: employeeTypeName => {
        switch (employeeTypeName) {
            case EmployeeTypeNames.SalesExecutive:
                return 'fa-phone';
                break;
            case EmployeeTypeNames.Manager:
                return 'fa-list-ol';
                break;
            case EmployeeTypeNames.Designer:
                return 'fa-paint-brush';
                break;
            case EmployeeTypeNames.Developer:
                return 'fa-code';
                break;
            case EmployeeTypeNames.LeadDeveloper:
                return 'fa-code-fork';
                break;
            case EmployeeTypeNames.ChiefTechnologyOfficer:
                return 'fa-black-tie';
                break;
            case EmployeeTypeNames.Researcher:
                return 'fa-flask';
                break;
            case EmployeeTypeNames.Marketer:
                return 'fa-newspaper-o';
                break;
            case EmployeeTypeNames.DevOps:
                return 'fa-server';
                break;
            case EmployeeTypeNames.HrManager:
                return 'fa-clock-o';
                break;
        }
    },
    CalculateTotalWorkHours: workingHours => {
        let end = workingHours.end.hour == 0 ? 24 : workingHours.end.hour;
        return end - workingHours.start.hour;
    },
    CalculateHourlyExhaustion: employee => {
        let wer = Helpers.CalculateWer();

        let baseDecrease = 0.7;
        let totalHours = Helpers.CalculateTotalWorkHours(employee.workingHours);

        // Long hours
        if(totalHours > 7) {
            baseDecrease += (totalHours - 7) * 0.2;
        }

        // If start hour not between 7-18
        if (!_.inRange(employee.workingHours.start.hour, 7, 18)) {
            baseDecrease += totalHours * 0.15;
        }

        // If end hour not between 7-18
        if (!_.inRange(employee.workingHours.end.hour, 7, 18)) {
            baseDecrease += totalHours * 0.10;
        }


        let decreaseAfterWerBonus = baseDecrease * (100 - wer.total) / 100;

        decreaseAfterWerBonus = Math.max(decreaseAfterWerBonus, 0.3);

        if (employee.requiredWer > wer.total) {
            decreaseAfterWerBonus = decreaseAfterWerBonus * 10;
        }

        return { base: baseDecrease, werBonus: wer.total, total: decreaseAfterWerBonus };
    }
};

let Cheats = {
    GetComponents: () => {
        let $rootScope = GetRootScope();
        Components.forEach(component => {
            $rootScope.settings.inventory[component.name] = 500;
        });
    },
    GetMoney: () => {
        GetRootScope().settings.balance = 1000000;
    },
    GetMaxTier: () => {
        GetRootScope().settings.xp = 500000;
    },
    CompleteAllTasks: () => {
        Helpers.GetAllEmployees().forEach(employee => {
            if (employee.task != null && employee.task.completedMinutes < employee.task.totalMinutes) {
                employee.task.completedMinutes = employee.task.totalMinutes;
            }
        });
    },
    SkipADay: () => {
        GetRootScope().settings.date = moment(GetRootScope().settings.date).add('day', 1).toDate();
        GetRootScope().$digest();
    },
};